<?php defined('BASEPATH') OR exit('No direct script access allowed');

class Install extends CI_Controller
{	
	public function __construct(){
		parent::__construct();
		$this->load->library(array('session','encrypt','auth'));
		$this->load->helper('url');
		$this->load->helper("form_helper");
		//check if the user is authorized
		$login_status = $this->session->userdata('is_loggedin');
		if(!isset($login_status) || $login_status !== 'true') {
			//if the user has authenticated via CAC or PIV, the controller will load 
			//if not then a 401 Unauthorized error will be shown
			if(!$this->auth->perform_auth()) { show_error('Unauthorized',401); }
		}
	}
	
	
	public function index() {
		$this->load_install_view(); //index view of install controller
	}
	
	/*This function attempts to create the database required for the API to run
	 */
	 public function create_db() {
		$sa_name = $this->input->post('sa_name',TRUE);
		$sa_pwd = $this->input->post('sa_pwd',TRUE);
		$sa_db_config = array(
			'hostname' => DATABASE_HOSTNAME,
			'username' => $sa_name,
			'password' => $sa_pwd,
			'database' => 'master',
			'dbdriver' => DATABASE_DRIVER,
		);
		$database = $this->load->database($sa_db_config, TRUE);
		//create database
		$database->query('CREATE DATABASE '.DATABASE_NAME);
		//set database permissions
		$database->query('USE ['.DATABASE_NAME.'] CREATE USER ['.DATABASE_USERNAME.'] FROM LOGIN ['.DATABASE_USERNAME.'] WITH DEFAULT_SCHEMA=[dbo]');
		$database->query("USE [".DATABASE_NAME."] EXEC sp_addrolemember 'db_datareader', " . $this->db->escape(DATABASE_USERNAME));
		$database->query("USE [".DATABASE_NAME."] EXEC sp_addrolemember 'db_datawriter', " . $this->db->escape(DATABASE_USERNAME));
		//create tables

		//users
		$database->query("
			USE [".DATABASE_NAME."]
			CREATE TABLE [dbo].[users](
				[user_id] [bigint] IDENTITY(1,1) NOT NULL,
				[username] [varchar](50) NOT NULL,
				[user_org_id] [bigint] NOT NULL,
				[user_created_time] [bigint] NOT NULL,
				[user_ext_mail] [nvarchar](max) NULL,
				[user_ep] [nvarchar](max) NOT NULL,
			PRIMARY KEY CLUSTERED
			(
				[user_id] ASC
			)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
			) ON [PRIMARY]
				
			ALTER TABLE [dbo].[users] ADD CONSTRAINT [unique_user_org_id] UNIQUE NONCLUSTERED
			(
				[user_org_id]
			)
		");
		
		//application request
		$database->query("
			USE [".DATABASE_NAME."]
			CREATE TABLE [dbo].[application_request](
				[id] [int] IDENTITY(1,1) NOT NULL,
				[name] [varchar](100) NULL,
				[requestor] [bigint] NOT NULL,
				[url] [varchar](1000) NULL,
				[description] [varchar](4000) NULL,
				[poc_name] [varchar](500) NULL,
				[poc_email] [varchar](100) NULL,
				[poc_phone] [varchar](20) NULL,
				[requested_date] [bigint] NULL,
				[approved_date] [bigint] NULL,
				[justification] [varchar](4000) NULL,
				[denial_reason] [varchar](4000) NULL,
				[denied] [tinyint] NULL,
			PRIMARY KEY CLUSTERED 
			(
				[id] ASC
			)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
			) ON [PRIMARY]
				
			ALTER TABLE [dbo].[application_request]  WITH CHECK ADD FOREIGN KEY([requestor])
			REFERENCES [dbo].[users] ([user_id])
		");
		//application
		$database->query("
			USE [".DATABASE_NAME."]
			CREATE TABLE [dbo].[application](
				[id] [int] IDENTITY(1,1) NOT NULL,
				[name] [varchar](100) NULL,
				[public_key] [varchar](100) NULL,
				[private_key] [varchar](100) NULL,
				[url] [varchar](1000) NULL,
				[description] [varchar](4000) NULL,
				[poc_name] [varchar](500) NULL,
				[poc_email] [varchar](100) NULL,
				[poc_phone] [varchar](20) NULL,
				[app_request_id] [int] NULL,
			PRIMARY KEY CLUSTERED 
			(
				[id] ASC
			)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
			) ON [PRIMARY]
			
			ALTER TABLE [dbo].[application]  WITH CHECK ADD FOREIGN KEY([app_request_id])
			REFERENCES [dbo].[application_request] ([id])
		");
		
		//request
		$database->query("
			USE [".DATABASE_NAME."]
			CREATE TABLE [dbo].[request](
				[id] [int] IDENTITY(1,1) NOT NULL,
				[application_id] [int] NULL,
				[call] [varchar](max) NULL,
				[call_date] [bigint] NULL,
				[response_code] [int] NULL,
				[response] [varchar](1000) NULL,
			PRIMARY KEY CLUSTERED 
			(
				[id] ASC
			)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
			) ON [PRIMARY]
			
			ALTER TABLE [dbo].[request]  WITH CHECK ADD FOREIGN KEY([application_id])
			REFERENCES [dbo].[application] ([id])
		");
		
		//account request
		$database->query("
			USE [".DATABASE_NAME."]
			CREATE TABLE [dbo].[account_request](
				[id] [int] IDENTITY(1,1) NOT NULL,
				[user_org_id] [bigint] NOT NULL,
				[first_name] [nvarchar](100) NOT NULL,
				[middle_name] [nvarchar](100) NULL,
				[last_name] [nvarchar](100) NOT NULL,
				[ext_mail] [nvarchar](200) NULL,
				[title] [nvarchar](300) NULL,
				[department] [nvarchar](300) NULL,
				[organization] [nvarchar](300) NULL,
				[location] [nvarchar](500) NULL,
				[telephone] [nvarchar](100) NULL,
				[mobile] [nvarchar](100) NULL,
				[request_date] [bigint] NOT NULL,
				[approved_date] [bigint] NULL,
				[justification] [varchar](4000) NULL,
				[denied] [tinyint] NOT NULL,
			PRIMARY KEY CLUSTERED 
			(
				[id] ASC
			)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
			) ON [PRIMARY]
		");
		
		//event log
		$database->query("
			USE [".DATABASE_NAME."]
			CREATE TABLE [dbo].[event_log](
				[id] [bigint] IDENTITY(1,1) NOT NULL,
				[target_type] [int] NOT NULL,
				[target_id] [bigint] NOT NULL,
				[actor_type] [int] NOT NULL,
				[actor_id] [bigint] NOT NULL,
				[action] [nvarchar] (MAX) NOT NULL,
				[event_date] [bigint] NOT NULL,
				[success] [tinyint] NOT NULL
			) ON [PRIMARY] TEXTIMAGE_ON [PRIMARY]
		");
		
		//logins
		$database->query("
			USE [".DATABASE_NAME."]
			CREATE TABLE [dbo].[logins](
				[id] [bigint] IDENTITY(1,1) NOT NULL,
				[session_id] [nvarchar](50) NOT NULL,
				[ip_address] [nvarchar](50) NOT NULL,
				[login_time] [bigint] NOT NULL,
				[success] [tinyint] NOT NULL,
				[error_msg] [nvarchar](MAX) NOT NULL,
				[org_id] [bigint] NOT NULL
			) ON [PRIMARY] TEXTIMAGE_ON [PRIMARY]
		");
		$database->query("
			USE [".DATABASE_NAME."]
			CREATE TABLE [dbo].[mail_log](
				[id] [bigint] IDENTITY(1,1) NOT NULL,
				[time] [bigint] NOT NULL,
				[size] [bigint] NOT NULL,
				[sender] [varchar](MAX) NOT NULL,
				[recipient] [varchar](MAX) NOT NULL,
				[attachment_types] [nvarchar](MAX) NOT NULL,
				[success] [tinyint] NOT NULL,
				[inbound_outbound] [tinyint] NOT NULL,
				[mdn] [tinyint] NULL
			) ON [PRIMARY] TEXTIMAGE_ON [PRIMARY]
		");
		
		//tickets
		$database->query("
			USE [".DATABASE_NAME."]
			CREATE TABLE [dbo].[ticket_category](
				[id] [bigint] IDENTITY(1,1) NOT NULL,
				[category] [varchar](200) NULL,
			PRIMARY KEY CLUSTERED
			(
				[id] ASC
			)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
			) ON [PRIMARY]
		");
		
		$query = $this->db->query("select * from ticket_category where category = 'Contact'");
		if ($query->num_rows() === 0)
			$this->db->query("insert into ticket_category (category) values ('Contact')");
		
		//tickets
		$database->query("
			USE [".DATABASE_NAME."]
			CREATE TABLE [dbo].[tickets](
				[id] [bigint] IDENTITY(1,1) NOT NULL,
				[parent_id] [bigint] NULL,
				[user_id] [bigint] NULL,
				[message] [varchar](4000) NULL,
				[open_date] [bigint] NULL,
				[close_date] [bigint] NULL,
				[category_id] [bigint] NULL,
			PRIMARY KEY CLUSTERED
			(
				[id] ASC
			)WITH (PAD_INDEX  = OFF, STATISTICS_NORECOMPUTE  = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS  = ON, ALLOW_PAGE_LOCKS  = ON) ON [PRIMARY]
			) ON [PRIMARY]
		
			ALTER TABLE [dbo].[tickets]  WITH CHECK ADD FOREIGN KEY([user_id])
			REFERENCES [dbo].[users] ([user_id])
				
			ALTER TABLE [dbo].[tickets]  WITH CHECK ADD FOREIGN KEY([parent_id])
			REFERENCES [dbo].[tickets] ([id])
				
			ALTER TABLE [dbo].[tickets]  WITH CHECK ADD FOREIGN KEY([category_id])
			REFERENCES [dbo].[ticket_category] ([id])
		");
		$this->session->set_flashdata('errors',sqlsrv_errors());
		redirect('install');
	 }
	 
	/* This function attempts to create the LDAP directory required for the API to run
	 */
	 public function create_ldap() {
		//if we get to this point, the configuration should be set enough to do a bind as the admin user
		$ldap_conn = $this->prepare_ldap_conn();
		$ldap_bind = @ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
		if($ldap_bind) {
			$dn_exp = ldap_explode_dn(LDAP_BASE_RDN,1);
			//base dn attributes
			$base_attrs = array(
				'objectClass' => array('dcObject','organization','top'),
				'dc' => $dn_exp[0],
				'o' => 'nodomain',
			);
			//api admins group attributes
			$api_admin_attrs = array(
				'objectClass' => 'groupOfNames',
				'member' => LDAP_ANON_ADMIN_USERNAME,
				'cn' => 'API Admins',
			);
			//direct api authorized apps group attributes
			$direct_api_apps_attrs = array(
				'objectClass' => 'groupOfNames',
				'member' => LDAP_ANON_ADMIN_USERNAME,
				'cn' => 'Direct API Enabled Applications',
			);
			//admin api authorized apps group attributes
			$admin_api_apps_attrs = array(
				'objectClass' => 'groupOfNames',
				'member' => LDAP_ANON_ADMIN_USERNAME,
				'cn' => 'Admin API Enabled Applications',
			);
			//apps group attributes
			$dn_exp = ldap_explode_dn(LDAP_APPLICATION_GROUP,1);
			$app_attrs = array(
				'objectClass' => array('organizationalUnit','top'),
				'ou' => $dn_exp[0],
			);
			//accounts group attributes
			$dn_exp = ldap_explode_dn(LDAP_ACCOUNT_GROUP,1);
			$account_attrs = array(
				'objectClass' => array('organizationalUnit','top'),
				'ou' => $dn_exp[0],
			);
			//disabled accounts group attributes
			$dn_exp = ldap_explode_dn(LDAP_DISABLED_ACCOUNT_GROUP,1);
			$disabeled_account_attrs = array(
				'objectClass' => array('organizationalUnit','top'),
				'ou' => $dn_exp[0],
			);
			//disabled applications group attributes
			$dn_exp = ldap_explode_dn(LDAP_DISABLED_APPLICATION_GROUP,1);
			$disable_app_attrs = array(
				'objectClass' => array('organizationalUnit','top'),
				'ou' => $dn_exp[0],
			);
			//groups group attributes
			$dn_exp = ldap_explode_dn(LDAP_GROUPS_GROUP,1);
			$groups_attrs = array(
				'objectClass' => array('organizationalUnit','top'),
				'ou' => $dn_exp[0],
			);
			//disabled groups group attributes
			$dn_exp = ldap_explode_dn(LDAP_DISABLED_GROUPS_GROUP,1);
			$disabeled_groups_attrs = array(
				'objectClass' => array('organizationalUnit','top'),
				'ou' => $dn_exp[0],
			);
			$add_base = ldap_add($ldap_conn,LDAP_BASE_RDN,$base_attrs);
			$add_api_admin = ldap_add($ldap_conn,LDAP_API_ADMIN_GROUP,$api_admin_attrs);
			$direct_api_apps = ldap_add($ldap_conn,LDAP_DIRECT_API_PERMISSIONS_GROUP,$direct_api_apps_attrs);
			$admin_api_apps = ldap_add($ldap_conn,LDAP_ADMIN_API_PERMISSIONS_GROUP,$admin_api_apps_attrs);
			$add_app = ldap_add($ldap_conn,LDAP_APPLICATION_GROUP,$app_attrs);
			$add_account = ldap_add($ldap_conn,LDAP_ACCOUNT_GROUP,$account_attrs);
			$add_disabled_app = ldap_add($ldap_conn,LDAP_DISABLED_APPLICATION_GROUP,$disable_app_attrs);
			$add_disabled_account = ldap_add($ldap_conn,LDAP_DISABLED_ACCOUNT_GROUP,$disabeled_account_attrs);
			$add_group = ldap_add($ldap_conn,LDAP_GROUPS_GROUP,$groups_attrs);
			$add_disabled_group = ldap_add($ldap_conn,LDAP_DISABLED_GROUPS_GROUP,$disabeled_groups_attrs);
		}
		if(!$ldap_bind || !$add_base || !$add_api_admin || !$add_app || !$add_account || !$add_disabled_app || !$add_disabled_account || !$direct_api_apps || !$admin_api_apps || !$add_group || !!$add_disabled_group) {
			if(ldap_error($ldap_conn) !== 'Success') { $this->session->set_flashdata('errors',ldap_error($ldap_conn)); }
		}
		redirect('install');
	 }
	public function import_users(){
		$data['title'] = 'VLER Direct API: Install';
		$this->load->view('api/header',$data);
		$this->output->append_output(form_open('/install/import_users_report'));
		$direct_name = $this->input->post('direct_name', TRUE);
		$ldap = $this->input->post('ldap', TRUE);
		$access = $this->input->post('access', TRUE);
		$direct_pwd = $this->input->post('direct_pwd', TRUE);
		if($direct_name){
			$sa_db_config = array(
				'hostname' => DATABASE_HOSTNAME,
				'username' => $direct_name,
				'password' => $direct_pwd,
				'database' => 'direct',
				'dbdriver' => DATABASE_DRIVER,
			);
		}
		else{
			$sa_db_config = array(
				'hostname' => DATABASE_HOSTNAME,
				'database' => 'direct',
				'dbdriver' => DATABASE_DRIVER,
			);
		}
		$sa_conn = $this->load->database($sa_db_config, TRUE);
		$results = $sa_conn->query('SELECT * FROM [dbo].[users] WHERE user_is_group=0');
		if($results){
			$results = $results->result();
			$this->output->append_output('<h2>Select users to Import</h2>');
			$this->output->append_output('<table class="display">');
			$this->output->append_output('<tr><th></th><th>Username</th><th>Org ID</th><th>Status</th></tr>');
			foreach($results as $result){
				if(USE_CAC_AUTH && USE_PIV_AUTH) {
					$org_id = (isset($result->user_edipi)) ? $result->user_edipi : $result->user_piv_id;
				}
				else if(USE_CAC_AUTH) {
					$org_id = $result->user_edipi;
				}
				else if(USE_PIV_AUTH) {
					$org_id = $result->user_piv_id;
				}
				else {
					//no authentication mechanism configured
				}
				$this->output->append_output('<tr><td><input type="checkbox" name="user_'.$result->user_name.'" id="user_'.$result->user_name.'" checked="true"/></td><td>'.$result->user_name.'</td><td>'.$org_id.'</td>');
				if($result->user_deleted_flag){
					$this->output->append_output('<td>Disabled</td>');
				}
				else{
					$this->output->append_output('<td>Enabled</td>');
				}
				$this->output->append_output('</tr>');
			}
			$this->output->append_output('</table>');
			$this->output->append_output('<input type="hidden" id="direct_name" name="direct_name" value="'.$direct_name.'"/>');
			$this->output->append_output('<input type="hidden" id="direct_pwd" name="direct_pwd" value="'.$direct_pwd.'"/>');
			$this->output->append_output('<input type="hidden" id="ldap" name="ldap" value="'.$ldap.'"/>');
			$this->output->append_output('<input type="hidden" id="access" name="access" value="'.$access.'"/>');
			$this->output->append_output('<div class="center"><input class="blue-button" type="submit" value="Submit"/></div>');
			$this->output->append_output(form_close());
			$this->output->append_output(form_open('/install/import_groups_report'));
			$this->output->append_output('<input type="hidden" id="ldap" name="ldap" value="'.$ldap.'"/>');
			$this->output->append_output('<input type="hidden" id="access" name="access" value="'.$access.'"/>');
			$this->output->append_output('<div class="center"><input class="blue-button" type="submit" value="Import Groups"/></div>');
			$this->output->append_output(form_close());
		}
		else{
			$this->output->append_output(form_close());
			$this->output->append_output('<span>Failed to get results</span>');
			$this->output->append_output(form_open('/install/import_users'));
			$this->output->append_output('<div class="form" style="float: none;">');
			$this->output->append_output('<label for="direct_name">Direct Database Username</label><input type="text" id="direct_name" name="direct_name" /><br /><br />');
			$this->output->append_output('<label for="direct_pwd">Direct Database Password</label><input type="password" id="direct_pwd" name="direct_pwd" /><br /><br />');
			$this->output->append_output('<label for="ldap">LDAP Base DN</label><input type="text" id="ldap" name="ldap" /><br /><br />');
			$this->output->append_output('</div>');
			$this->output->append_output('<div class="center"><input class="blue-button" type="submit" value="Import Users"/></div>');
			$this->output->append_output(form_close());
		}
		$this->output->append_output('<br/><a href="/install">Return</a>');
	}
	
	public function import_users_report(){
		$data['title'] = 'VLER Direct API: Install';
		$this->load->view('api/header',$data);
		$direct_name = $this->input->post('direct_name', TRUE);
		$ldap = $this->input->post('ldap', TRUE);
		$access = $this->input->post('access', TRUE);
		$direct_pwd = $this->input->post('direct_pwd', TRUE);
		if($direct_name){
			$sa_db_config = array(
				'hostname' => DATABASE_HOSTNAME,
				'username' => $direct_name,
				'password' => $direct_pwd,
				'database' => 'direct',
				'dbdriver' => DATABASE_DRIVER,
			);
		}
		else{
			$sa_db_config = array(
				'hostname' => DATABASE_HOSTNAME,
				'database' => 'direct',
				'dbdriver' => DATABASE_DRIVER,
			);
		}
		$sa_conn = $this->load->database($sa_db_config, TRUE);
		$results = $sa_conn->query('SELECT * FROM [dbo].[users] WHERE user_is_group=0');
		if($results){
			$results = $results->result();
			$this->db = NULL;
			$this->load->library(array('eventlog'));
			$this->load->model('usersmodel');
			$this->load->model('accountrequestmodel');
			$user_id = $this->usersmodel->get_user_id_from_org_id($this->encrypt->decode($this->session->userdata('user_id')));
			$ldap_conn = $this->prepare_ldap_conn();
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
			if($ldap_bind) {
				$fields = array('uid','cn','givenname','initials','sn','title','departmentnumber','o','physicaldeliveryofficename','telephonenumber','mobile','mail','','gidnumber','uidnumber','homedirectory');
				$this->output->append_output('<h2>Found '.count($results).' user(s)</h2>');
				$this->output->append_output('<table style="background:lightgray;">');
				$added = $failed = $exist = $inuser = $ignored = 0;
				foreach ($results as $user){
					if(USE_CAC_AUTH && USE_PIV_AUTH) { $org_id = (isset($user->user_edipi)) ? $user->user_edipi : $user->user_piv_id; }
					else if(USE_CAC_AUTH) { $org_id = $user->user_edipi; }
					else if(USE_PIV_AUTH) { $org_id = $user->user_piv_id; }
					else {
						//no authentication mechanism configured
					}
					
					if($user->user_deleted_flag){ $base_dn = "ou=deletedaccounts,".$ldap; }
					else { $base_dn = "ou=accounts,".$ldap; }
					
					$this->output->append_output('<tr><td>'.$user->user_name.'</td>');
					if(!$this->input->post('user_'.$user->user_name, TRUE)) {
						$this->output->append_output('<td> was ignored</td>');
						$ignored += 1;
					}
					else {
						$search = @ldap_search($ldap_conn, $base_dn, '(&(uid='.$user->user_name.'))', $fields);
						if($search){ $entries = ldap_get_entries($ldap_conn, $search); }
						else { $entries = array('count' => 0); }
						
						if(!isset($org_id) || !$org_id){ //No Organizational ID set
							if(USE_CAC_AUTH && USE_PIV_AUTH) { 
								$this->output->append_output('<td style="color:brown"> User has no EDIPI or PIV ID, possible install user (not added)</td>');
							}
							else if(USE_CAC_AUTH) { 
								$this->output->append_output('<td style="color:brown"> User has no EDIPI configured, possible PIV user (not added)</td>');
							}
							else if(USE_PIV_AUTH) { 
								$this->output->append_output('<td style="color:brown"> User has no PIV ID configured, possible CAC user (not added)</td>');
							}
							else { 
								$this->output->append_output('<td style="color:brown"> No authentication mechanism chosen, no users will be added (not added)</td>');
							}
						
							$inuser += 1;	
						}
						else {
							if($this->usersmodel->org_id_exists($org_id) || $this->usersmodel->username_exists($user->user_name)){
								$exist += 1;
								if($this->encrypt->decode($this->session->userdata('user_id')) === $org_id){
									$this->output->append_output('<td style="color:blue"> current user');
								}
								else{
									$this->output->append_output('<td style="color:blue"> username or id already exist');
								}
								if(isset($access) && $access !== ''){
									$target_id = $this->usersmodel->get_user_id_from_org_id($org_id);
									$response = $this->usersmodel->change_group_membership('add',$target_id,$access);
									$this->eventlog->create_event(3, $target_id, 3, $user_id, 'Access added '.$access , time(), $response);
									if($response){
										$this->output->append_output(' added to group');
									}
									else{
										$this->output->append_output('<span style="color:red"> did not add to group</span>');
									}
								}
								$this->output->append_output('</td>');
							}
							else{
								if($entries['count']){
									$userattribute = array();
									foreach($fields as $key) {
										if(isset($entries[0][$key][0])) {
											$userattribute[$key] = $entries[0][$key][0];
										} 
									}
									$userattribute['objectClass'] = array('posixAccount', 'top', 'person', 'organizationalPerson', 'inetOrgPerson');
									$created = $this->usersmodel->create_user($user->user_name, 1, $org_id, $user->user_mail, $userattribute);
									if ($created){
										$added += 1;
										$this->output->append_output('<td style="color:green"> was added');
										$target_id = $this->usersmodel->get_user_id_from_org_id($user->user_edipi);
										$this->eventlog->create_event(3, $target_id, 3,$user_id, 'User imported', time(), 1);
										if(!$this->accountrequestmodel->create_request((isset($userattribute['givenname'])?$userattribute['givenname']:""), (isset($userattribute['initials'])?$userattribute['initials']:''), (isset($userattribute['sn'])?$userattribute['sn']:''), $user->user_mail, (isset($userattribute['title'])?$userattribute['title']:''), (isset($userattribute['departmentnumber'])?$userattribute['departmentnumber']:''), (isset($userattribute['o'])?$userattribute['o']:''), (isset($userattribute['telephonenumber'])?$userattribute['telephonenumber']:''), (isset($userattribute['mobile'])?$userattribute['mobile']:''), (isset($userattribute['physicaldeliveryofficename'])?$userattribute['physicaldeliveryofficename']:'') , $org_id)){
											$this->output->append_output('<span style="red"> |error could not create request|</span>');
										}
										else{
											
											$request_id=$this->accountrequestmodel->get_ids($org_id);
											if($request_id){
												$request_id = $request_id->result();
												if(!$this->accountrequestmodel->approve_request($request_id[0]->id)){
													$this->output->append_output('<span style="red"> |failed to approve request|</span>');
												}
											}
											else{
												$this->output->append_output('<span style="red"> |failed to find request|</span>');
											}
										}
										
										if($user->user_deleted_flag){
											if($this->usersmodel->disable_user($target_id)){
												$this->eventlog->create_event(3, $target_id, 3, $user_id, "Disable account", time(), 1);
												$this->output->append_output(' and was disabled');
											}
											else{
												$this->eventlog->create_event(3, $target_id, 3, $user_id, "Disable account", time(), 0);
												$this->output->append_output('<span style="red"> but was not disabled</span>');
											}
										}
										if(isset($access) && $access !== ""){
											$response = $this->usersmodel->change_group_membership('add',$target_id,$access);
											$this->eventlog->create_event(3, $target_id, 3, $user_id, 'Access added '.$access , time(), $response);
											if($response){
												$this->output->append_output(' added to group');
											}
											else{
												$this->output->append_output('<span style="color:red"> did not add to group</span>');
											}
										}
										$this->output->append_output('</td>');
									}
									else{
										$failed += 1;
										$this->output->append_output('<td style="color:red"> failed to find create user</td>');
									}
								}
								else{
									$failed += 1;
									$this->output->append_output('<td style="color:red"> failed to find entry in LDAP</td>');
								}
							}
						}	
					}
					$this->output->append_output('</tr>');
				}
				$this->output->append_output('</table>');
				$this->output->append_output('<br/><h3>Report:</h3>');
				$this->output->append_output('<table style="background: lightblue;">');
				$this->output->append_output('<tr><td>Added: </td><td style="color:green;">'.$added.'</td></tr>');
				$this->output->append_output('<tr><td>Failed: </td><td style="color:red;">'.$failed.'</td></tr>');
				$this->output->append_output('<tr><td>Existing: </td><td style="color:blue;">'.$exist.'</td></tr>');
				$this->output->append_output('<tr><td>Install User:</td><td style="color:brown;">'.$inuser.'</td></tr>');
				$this->output->append_output('<tr><td>Ignored:</td><td>'.$ignored.'</td></tr>');
				$this->output->append_output('</table>');
			}
			else{
				$this->output->append_output('<span>Failed to connect to LDAP</span>');
			}
		}
		else {
			$this->output->append_output('<span>Failed to get results</span>');
			$this->output->append_output(form_open('/install/import_users')."\n");
			$this->output->append_output('<div class="form" style="float: none;">');
			$this->output->append_output('<label for="direct_name">Direct Database Username</label><input type="text" id="direct_name" name="direct_name" /><br /><br />');
			$this->output->append_output('<label for="direct_pwd">Direct Database Password</label><input type="password" id="direct_pwd" name="direct_pwd" /><br /><br />');
			$this->output->append_output('<label for="ldap">LDAP Base DN</label><input type="text" id="ldap" name="ldap" /><br /><br />');
			$this->output->append_output('</div>');
			$this->output->append_output('<div class="center"><input class="blue-button" type="submit" value="Import Users"/></div>');
			$this->output->append_output(form_close());
		}
		$this->output->append_output('<br/><a href="/install">Return</a>');
	}
	
	public function import_groups_report(){
		$access = $this->input->post('access', TRUE);
		$data['title'] = 'VLER Direct API: Install';
		$this->load->view('api/header',$data);
		$ldap = $this->input->post('ldap', TRUE);
		$this->load->library(array("eventlog"));
		$this->load->model('usersmodel');
		$this->load->model('groupsmodel');
		$user_id = $this->usersmodel->get_user_id_from_org_id($this->encrypt->decode($this->session->userdata('user_id')));
		$ldap_conn = $this->prepare_ldap_conn();
		$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if($ldap_bind) {
			$fields = array('cn','ou','description');
			$added = 0;
			$failed = 0;
			$exist = 0;
			$ignored = 0;
			$base_dn = "ou=groups,".$ldap;
			$search = @ldap_search($ldap_conn, $base_dn, '(&(cn=*))', $fields);	
			$this->output->append_output("<p>Groups</p>");	
			$this->output->append_output('<table style="background:lightgray;">');			
			if($search){
				$entries = ldap_get_entries($ldap_conn, $search);
			}
			else{
				$entries = array('count' => 0);
			}
			for($count = 0; $count < $entries['count'] ; $count++){
				$this->output->append_output('<tr><td>'.$entries[$count]['ou'][0].'</td>');
				if($this->usersmodel->username_exists($entries[$count]['ou'][0])){
					$this->output->append_output('<td style="color:blue">Username exists');
					$exist += 1;
					if(isset($access) && $access !== ""){
						$response = $this->groupsmodel->change_group_membership("add",$entries[$count]['ou'][0],$access);
						$this->eventlog->create_event( 6,0,3 ,$user_id, "Access added for ".$entries[$count]['ou'][0].": ".$access , time(), $response);
						if($response){
							$this->output->append_output(' added to group');
						}
						else{
							$this->output->append_output('<span style="color:red"> did not add to group</span>');
						}
					}
					$this->output->append_output('</td>');
				}
				else{
					if(!isset($entries[$count]['description'])){
						$entries[$count]['description'] = array("Imported Group");
					}
					if($this->groupsmodel->create_group($entries[$count]['ou'][0],$entries[$count]['cn'][0],$entries[$count]['description'][0])){
						$this->output->append_output('<td style="color:green">was added');
						$added += 1;
						if(isset($access) && $access !== ""){
							$response = $this->groupsmodel->change_group_membership("add",$entries[$count]['ou'][0],$access);
							$this->eventlog->create_event( 6,0,3 ,$user_id, "Access added for ".$entries[$count]['ou'][0].": ".$access , time(), $response);
							if($response){
								$this->output->append_output(' added to group');
							}
							else{
								$this->output->append_output('<span style="color:red"> did not add to group</span>');
							}
						}
						$this->output->append_output('</td>');
						$this->eventlog->create_event(6, 0, 3,$user_id, "Imported Group: ".$entries[$count]['ou'][0], time(), 1);
					}
					else{
						$this->output->append_output('<td style="color:red">FAILED</td>');
						$failed += 1;
						$this->eventlog->create_event(6, 0, 3,$user_id, "Imported Group: ".$entries[$count]['ou'][0], time(), 0);
					}
				}
				$this->output->append_output('</tr>');
			}
			$this->output->append_output('</table>');
			$this->output->append_output('<br/><h3>Report:</h3>');
			$this->output->append_output('<table style="background: lightblue;">');
			$this->output->append_output('<tr><td>Added: </td><td style="color:green;">'.$added.'</td></tr>');
			$this->output->append_output('<tr><td>Failed: </td><td style="color:red;">'.$failed.'</td></tr>');
			$this->output->append_output('<tr><td>Existing: </td><td style="color:blue;">'.$exist.'</td></tr>');
			$this->output->append_output('<tr><td>Ignored:</td><td>'.$ignored.'</td></tr>');
			$this->output->append_output('</table>');
			$this->output->append_output("<p>Deleted Groups</p>");	
			$this->output->append_output('<table style="background:lightgray;">');
			$added = 0;
			$failed = 0;
			$exist = 0;
			$ignored = 0;
			$base_dn = "ou=deletedgroups,".$ldap;
			$search = @ldap_search($ldap_conn, $base_dn, '(&(cn=*))', $fields);	
				
			if($search){
				$entries = ldap_get_entries($ldap_conn, $search);
			}
			else{
				$entries = array('count' => 0);
			}
			for($count = 0; $count < $entries['count'] ; $count++){
				$this->output->append_output('<tr><td>'.$entries[$count]['ou'][0].'</td>');
				if($this->usersmodel->username_exists($entries[$count]['ou'][0])){
					$this->output->append_output('<td style="color:blue">Username exists');
					$exist += 1;
					if(isset($access) && $access !== ""){
						$response = $this->groupsmodel->change_group_membership("add",$entries[$count]['ou'][0],$access);
						$this->eventlog->create_event( 6,0,3 ,$user_id, "Access added for ".$entries[$count]['ou'][0].": ".$access , time(), $response);
						if($response){
							$this->output->append_output(' added to group');
						}
						else{
							$this->output->append_output('<span style="color:red"> did not add to group</span>');
						}
					}
					$this->output->append_output('</td>');
				}
				else{
					if(!isset($entries[$count]['description'])){
						$entries[$count]['description'] = array("Imported Group");
					}
					if($this->groupsmodel->create_group($entries[$count]['ou'][0],$entries[$count]['cn'][0],$entries[$count]['description'][0])){
						$this->output->append_output('<td style="color:green">was added');
						$added += 1;
						$this->eventlog->create_event(6, 0, 3,$user_id, "Imported Group: ".$entries[$count]['ou'][0], time(), 1);
						if($this->groupsmodel->disable_group($entries[$count]['ou'][0])){
							$this->output->append_output(' and disabled');
							$this->eventlog->create_event(6, 0, 3,$user_id, "Disable group: ".$entries[$count]['ou'][0], time(), 1);
						}
						else{
							$this->output->append_output('<span="color:red"> FAILED TO DISABLE');
							$this->eventlog->create_event(6, 0, 3,$user_id, "Disable group: ".$entries[$count]['ou'][0], time(), 0);
						}
						if(isset($access) && $access !== ""){
							$response = $this->groupsmodel->change_group_membership("add",$entries[$count]['ou'][0],$access);
							$this->eventlog->create_event( 6,0,3 ,$user_id, "Access added for ".$entries[$count]['ou'][0].": ".$access , time(), $response);
							if($response){
								$this->output->append_output(' added to group');
							}
							else{
								$this->output->append_output('<span style="color:red"> did not add to group</span>');
							}
						}
						if(isset($access) && $access !== ""){
							$response = $this->groupsmodel->change_group_membership("add",$entries[$count]['ou'][0],$access);
							$this->eventlog->create_event( 6,0,3 ,$user_id, "Access added for ".$entries[$count]['ou'][0].": ".$access , time(), $response);
							if($response){
								$this->output->append_output(' added to group');
							}
							else{
								$this->output->append_output('<span style="color:red"> did not add to group</span>');
							}
						}
						$this->output->append_output('</td>');
					}
					else{
						$this->output->append_output('<td style="color:red">FAILED</td>');
						$failed += 1;
						$this->eventlog->create_event(6, 0, 3,$user_id, "Imported Group: ".$entries[$count]['ou'][0], time(), 0);
					}
				}
				$this->output->append_output('</tr>');
			}
			$this->output->append_output('</table>');	
			$this->output->append_output('<br/><h3>Report:</h3>');
			$this->output->append_output('<table style="background: lightblue;">');
			$this->output->append_output('<tr><td>Added: </td><td style="color:green;">'.$added.'</td></tr>');
			$this->output->append_output('<tr><td>Failed: </td><td style="color:red;">'.$failed.'</td></tr>');
			$this->output->append_output('<tr><td>Existing: </td><td style="color:blue;">'.$exist.'</td></tr>');
			$this->output->append_output('<tr><td>Ignored:</td><td>'.$ignored.'</td></tr>');
			$this->output->append_output('</table>');
		}
		else{
			$this->output->append_output('<span>Failed to connect to LDAP</span>');
		}
		
		$this->output->append_output('<br/><a href="/install">Return</a>');
	}
	/* This function attempts to create the API application and first user records
	 */
	 public function create_user() {
		$this->load->model('usersmodel');
		$this->load->model('applicationmodel');
		$this->load->model('applicationrequestmodel');
		$this->load->model('accountrequestmodel');
		//get form data
		$first = $this->input->post('first_name', TRUE);
		$middle = $this->input->post('middle_name', TRUE);
		$last = $this->input->post('last_name', TRUE);
		$mail = $this->input->post('ext_mail', TRUE);
		$title = $this->input->post('title', TRUE);
		$department = $this->input->post('department', TRUE);
		$organization = $this->input->post('organization', TRUE);
		$telephone = $this->input->post('telephone', TRUE);
		$mobile = $this->input->post('mobile', TRUE);
		$location = $this->input->post('location', TRUE);
		$user_org_id = $this->encrypt->decode($this->session->userdata('user_id'));
		$username = $this->get_new_username($first,$last);
		
		//create app info
		$public = hash('sha256', openssl_random_pseudo_bytes(32));
		$private = hash('sha256', openssl_random_pseudo_bytes(32));
		$name = 'VLER Direct API';
		$url = 'https://'.API_DOMAIN;
		$desc = 'VLER Direct API';
		$just = 'Default Application';
		$poc_name = $first.' ' .$last;
		$poc_email = $mail;
		$poc_phone = $telephone;
		
		
		//create application
		if($this->applicationmodel->create_application($public, $private, $name, null, $url, $desc, $poc_name, $poc_email, $poc_phone, null)) {
			$app = $this->applicationmodel->get_application_from_public($public);
			$row = $app->row_array(0);
			$app_id = $row['id'];
			
			//create user
			$user_attrs = array(
				'objectClass' => array('posixAccount', 'top', 'person', 'organizationalPerson', 'inetOrgPerson'),
				'gidNumber' => '5000',
				'uidNumber' => '5000',
				'uid' => $username,
				'homeDirectory' => '/var/mailboxes/'.$username,
				'mail' => $username.'@'.DIRECT_DOMAIN, //don't confuse this with external mail, this is internal direct address
				'givenName' => $first,
				'sn' => $last,
				'cn' => $first.' '.$last,
			);
			if($middle){
				$user_attrs['initials'] = $middle;
			}
			if($title){
				$user_attrs['title'] = $title;
			}
			if($mobile){
				$user_attrs['mobile'] = $mobile;
			}
			if($telephone){
				$user_attrs['telephoneNumber'] = $telephone;
			}
			if($organization){
				$user_attrs['o'] = $organization;
			}
			if($department){
				$user_attrs['departmentNumber'] = $department;
			}
			if($location){
				$user_attrs['physicalDeliveryOfficeName'] = $location;
			}
			$this->usersmodel->create_user($username, $app_id, $user_org_id, $mail, $user_attrs);
			$this->accountrequestmodel->create_request((isset($user_attrs['givenName'])?$user_attrs['givenName']:""), (isset($user_attrs['initials'])?$user_attrs['initials']:""), (isset($user_attrs['sn'])?$user_attrs['sn']:""), $mail, (isset($user_attrs['title'])?$user_attrs['title']:""), (isset($user_attrs['departmentNumber'])?$user_attrs['departmentNumber']:""), (isset($user_attrs['o'])?$user_attrs['o']:""), (isset($user_attrs['telephoneNumber'])?$user_attrs['telephoneNumber']:""), (isset($user_attrs['mobile'])?$user_attrs['mobile']:""), (isset($user_attrs['physicalDeliveryOfficeName'])?$user_attrs['physicalDeliveryOfficeName']:"") , $user_org_id);
			$r_id=$this->accountrequestmodel->get_ids($user_org_id);
			if($r_id){
				$r_id = $r_id->result();
				$this->accountrequestmodel->approve_request($r_id[0]->id);
			}
			else{
				$this->accountrequestmodel->approve_request(1);//assume first ID
			}
			//create application request
			$this->applicationrequestmodel->create_request($name, $this->usersmodel->get_user_id_from_org_id($user_org_id), $url, $desc, $just, $poc_name, $poc_email, $poc_phone);
			$requests = $this->applicationrequestmodel->get_ids($name);
			$request_row = $requests->row_array(0);
			$request_id = $request_row['id'];
			//approve request
			$this->applicationrequestmodel->approve_request($request_id);
			//update app
			$undo_query = $this->db->query('UPDATE application SET app_request_id = '.$request_id.' WHERE id = '.$app_id);
			//add user to API admins / default app admins group
			//if we get to this point, the configuration should be set enough to do a bind as the admin user
			$ldap_conn = $this->prepare_ldap_conn();
			$ldap_bind = @ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
			if($ldap_bind) {
				ldap_mod_add($ldap_conn,LDAP_API_ADMIN_GROUP,array('member' => $this->usersmodel->get_dn_from_username($username))); 
				ldap_mod_add($ldap_conn,$this->applicationmodel->get_admins_dn_from_app_id($app_id),array('member' => $this->usersmodel->get_dn_from_username($username)));
				ldap_mod_add($ldap_conn,LDAP_DIRECT_API_PERMISSIONS_GROUP,array('member' => $this->applicationmodel->get_dn_from_app_id($app_id)));
				ldap_mod_add($ldap_conn,LDAP_ADMIN_API_PERMISSIONS_GROUP,array('member' => $this->applicationmodel->get_dn_from_app_id($app_id)));
			}
		}
		//TO-DO: better error reporting if something goes wrong with this, need to display LDAP erros too
		$this->session->set_flashdata('errors',sqlsrv_errors());
		redirect('install');
	 }
	 
	/* -----------------------------*
	 *  PRIVATE FUNCTIONS           *
	 * -----------------------------*/
	 
	 /* Normally this would be done as an actual view, but in an attempt to make the install
	  * process contained to a single file, we construct the view here.
	  */
	 private function load_install_view() {
		$data['title'] = 'VLER Direct API: Install';
		$this->load->view('api/header',$data);
		$skip_data = $this->input->post('skip_data',TRUE);
		$skip_ldap = $this->input->post('skip_ldap',TRUE);
		$skip_user = $this->input->post('skip_user',TRUE);
		$this->output->append_output('<div class="column">');
		//if any errors appear in the flashdata, display them before the install header
		if(is_array($this->session->flashdata('errors'))) {
			$errors = $this->session->flashdata('errors');
			$this->output->append_output('<div class="message_error">');
			foreach($errors as $error) { $this->output->append_output($error['message']); }
			$this->output->append_output('</div>');
		}
		else if(strlen($this->session->flashdata('errors'))) {
			$errors = $this->session->flashdata('errors');
			$this->output->append_output('<div class="message_error">');
			$this->output->append_output($errors);
			$this->output->append_output('</div>');
		}
		$this->output->append_output('</div>');
		
		//Install header
		$this->output->append_output('<h1>Install</h1>');
		$this->output->append_output('<div class="column">');
		
		//check configuration
		$config_errs = $this->check_ci_configuration();
		if(strlen($config_errs['message']) > 0) {
			$this->output->append_output('<div class="message_error">Please resolve the following configuration errors before doing install: <br /><ul>' . $config_errs['message'] . '</ul></div>');
		}
		
		if($config_errs['status'] === TRUE) {			
			//Database
			$db_status = $this->check_db_configuration();
			if(isset($skip_data) && $skip_data === "Skip"){
				$db_status['status'] = TRUE;
			}
			if($db_status['status'] === FALSE) {
				$this->output->append_output('<h2>Database</h2>');
				if(strlen($db_status['message']) > 0) { $this->output->append_output($db_status['message']); }
				$this->output->append_output('<div class="body">');
				//if database has not been set up properly yet, display database creation form
				if($db_status['status'] === FALSE) {
					$this->output->append_output(form_open('/install/create_db'));
					$this->output->append_output('<div class="form" style="float: none;">');
					$this->output->append_output('<label for="sa_name">Database SA</label><input type="text" id="sa_name" name="sa_name" /><br /><br />');
					$this->output->append_output('<label for="sa_pwd">Database SA Password</label><input type="password" id="sa_pwd" name="sa_pwd" />');
					$this->output->append_output('</div>');
					$this->output->append_output('<div class="center"><input class="blue-button" type="submit" value="Create Database"/></div>');
					$this->output->append_output(form_close());
					$this->output->append_output(form_open('/install'));
					if(isset($skip_ldap) && $skip_ldap === "Skip"){
						$this->output->append_output('<input type="hidden" id="skip_ldap" name="skip_ldap" value="Skip">');
					}
					$this->output->append_output('<input type="hidden" id="skip_data" name="skip_data" value="Skip">');
					$this->output->append_output('<input type="submit" value="Skip Database"/>');
					$this->output->append_output(form_close());
				}
			}
			else {
				$this->output->append_output('<h2>Database <span style="color: green;">&#x2713;</span></h2>');
			}
			//LDAP
			$ldap_status = $this->check_ldap_configuration();
			if(isset($skip_ldap) && $skip_ldap === "Skip"){
					$ldap_status['status'] = TRUE;
			}
			if($ldap_status['status'] === FALSE) {
				$this->output->append_output('<h2>LDAP</h2>');
				if(strlen($ldap_status['message']) > 0) { $this->output->append_output($ldap_status['message']); }
				if($ldap_status['status'] === FALSE) {
					$this->output->append_output(form_open('/install/create_ldap'));
					$this->output->append_output('<div class="center"><input class="blue-button" type="submit" value="Create LDAP Directory"/></div>');
					$this->output->append_output(form_close());
					$this->output->append_output(form_open('/install'));
					if(isset($skip_ldap) && $skip_ldap === "Skip"){
						$this->output->append_output('<input type="hidden" id="skip_data" name="skip_data" value="Skip">');
					}
					$this->output->append_output('<input type="hidden" id="skip_ldap" name="skip_ldap" value="Skip">');
					$this->output->append_output('<input type="submit" value="Skip LDAP Creation"/>');
					$this->output->append_output(form_close());
				}
			}
			else {
				$this->output->append_output('<h2>LDAP <span style="color: green;">&#x2713;</span></h2>');
			}
			if($db_status['status'] !== FALSE && $ldap_status['status'] !== FALSE) {
				$user_status = $this->check_user_configuration();
				if(isset($skip_user) && $skip_user === "Skip"){
					$user_status['status'] = TRUE;
				}
				if($user_status['status'] === FALSE) {
					$user_info = $this->session->userdata('user_info');
					$this->output->append_output('<h2>Create Initial User</h2>');
					if(strlen($user_status['message']) > 0) { $this->output->append_output($user_status['message']); }
					$this->output->append_output(form_open('/install/create_user'));
					$this->output->append_output('<div class="form">');
					$this->output->append_output('<label for="first_name">First Name:</label> <input type="text" name="first_name" id="first_name" value="'.$user_info['first_name'].'" />');
					$this->output->append_output('<label for="middle_name">Middle Name:</label> <input type="text" name="middle_name" id="middle_name" value="'.$user_info['middle_name'].'" />');
					$this->output->append_output('<label for="last_name">Last Name:</label> <input type="text" name="last_name" id="last_name" value="'.$user_info['last_name'].'" />');
					$this->output->append_output('<label for="ext_mail">Email Address:</label> <input type="text" name="ext_mail" id="ext_mail" value="'.$user_info['email'].'" />');
					$this->output->append_output('<label for="title">Title:</label> <input type="text" name="title" id="title"/>');
					$this->output->append_output('<label for="department">Department:</label> <input type="text" name="department" id="department"/>');
					$this->output->append_output('<label for="organization">Organization:</label> <input type="text" name="organization" id="organization"/>');
					$this->output->append_output('<label for="telephone">Telephone:</label> <input type="text" name="telephone" id="telephone"/>');
					$this->output->append_output('<label for="mobile">Mobile Phone:</label> <input type="text" name="mobile" id="mobile"/>');
					$this->output->append_output('<label for="location">Location:</label> <input type="text" name="location" id="location"/>');
					$this->output->append_output('</div>');
					$this->output->append_output('<div class="center"><input class="blue-button" type="submit" value="Save"/></div>');
					$this->output->append_output(form_close());
					$this->output->append_output(form_open('/install'));
					$this->output->append_output('<input type="hidden" id="skip_data" name="skip_data" value="Skip">');
					$this->output->append_output('<input type="hidden" id="skip_ldap" name="skip_ldap" value="Skip">');
					$this->output->append_output('<input type="hidden" id="skip_user" name="skip_user" value="Skip">');
					$this->output->append_output('<input type="submit" value="Skip User Creation"/>');
					$this->output->append_output(form_close());
				}
				else {
					$this->output->append_output('<h2>Create Initial User <span style="color: green;">&#x2713;</span></h2>');
					$this->output->append_output('<p>Install has been completed, delete file applications/controllers/install.php</p>');
					
					$this->load->model('applicationmodel');
					$app = $this->applicationmodel->get_application(1);
					$app_arr = $app->row_array(0);
					$this->output->append_output('<p>
													Replace WEBSERVICE_PUBLIC_KEY and WEBSERVICE_PRIVATE KEY constants in application/config/constants.php to these values: <br /><br />
													define(\'WEBSERVICE_PUBLIC_KEY\', \''.$app_arr['public_key'].'\');<br />
													define(\'WEBSERVICE_PRIVATE_KEY\', \''.$app_arr['private_key'].'\');
												  </p>
												');
					$this->output->append_output('<h2>Import Users From Webmail</h2>');
					$this->output->append_output(form_open('/install/import_users'));
					$this->output->append_output('<div class="form" style="float: none;">');
					$this->output->append_output('<label for="ldap">LDAP Base DN</label><input type="text" id="ldap" name="ldap" />');
					$this->output->append_output('<label for="access">Added access to</label><input type="text" id="access" name="access" />');
					$this->output->append_output('</div>');
					$this->output->append_output('<div class="center"><input class="blue-button" type="submit" value="Import Users"/></div>');
					$this->output->append_output(form_close());
				}
			}
			$this->output->append_output('</div>');
		
		}
		$this->output->append_output('</div>');
	 }
	 
	 /* This function checks if the items defined in the code igniter configuration are working properly 
	  */
	 private function check_ci_configuration() {
		$status = TRUE;
		$output = '';
		//test constant definitions
		$constants = array(
			'DIRECT_DOMAIN','API_DOMAIN', 'API_ADMINPANEL_DOMAIN',
			'GATEWAY_SMTP_HOSTNAME','GATEWAY_SMTP_PORT', 'GATEWAY_SMTP_TIMEOUT',
			'DIRECT_SEND_PROTOCOL', 'DIRECT_ATTACHMENT_SIZE_LIMIT',
			'REPORT_LOGS_PER_PAGE_COUNT',
			'DATABASE_HOSTNAME', 'DATABASE_NAME', 'DATABASE_USERNAME', 'DATABASE_PASSWORD', 'DATABASE_DRIVER',
			'LDAP_HOSTNAME', 'LDAP_PORT', 'LDAP_BASE_RDN',
			'LDAP_API_ADMIN_GROUP', 'LDAP_APPLICATION_GROUP', 'LDAP_DISABLED_APPLICATION_GROUP', 'LDAP_APPLICATION_ADMIN_GROUP_NAME', 'LDAP_APPLICATION_USER_GROUP_NAME', 'LDAP_ACCOUNT_GROUP', 'LDAP_DISABLED_ACCOUNT_GROUP', 'LDAP_GROUPS_GROUP', 'LDAP_DISABLED_GROUPS_GROUP', 
			'LDAP_DIRECT_API_PERMISSIONS_GROUP','LDAP_ADMIN_API_PERMISSIONS_GROUP',
			'LDAP_ANON_SEARCH_USERNAME','LDAP_ANON_SEARCH_PASSWORD','LDAP_ANON_ADMIN_USERNAME', 'LDAP_ANON_ADMIN_PASSWORD',
			'WEBSERVICE_URL', 'WEBSERVICE_PUBLIC_KEY', 'WEBSERVICE_PRIVATE_KEY',
		);
		
		foreach($constants as $constant) {
			if(!defined($constant)) { 
				$output .= '<li>' . $constant . ' is not defined in config/constants.php</li>';
				$status = FALSE;
			}
		}
		
		//test database connection
		$db = $this->load->database('default',TRUE);
		if(FALSE === $db->conn_id) {
			$error_output = '';
			$errors = sqlsrv_errors();
			foreach($errors as $error) { $error_output .= $error['message']; }
			preg_match_all("/Cannot open database \"(.*)?\" requested by the login/",$error_output,$matches);
			//if the database driver couldn't load the database specified in configuration,
			//don't display a config warning because we create it during the installation process
			if($matches[1][0] !== DATABASE_NAME) {
				$output .= '<li>Could not connect to database using settings in config/constants.php</li>';
				$output .= $error_output;
				$status = FALSE;
			}
		}
		//test LDAP connection
		$ldap_conn = $this->prepare_ldap_conn();
		if(FALSE === $ldap_conn) { //this will apparently never happen because ldap_connect is a stupid function, but check anyway
			$output .= '<li>Could not connect to LDAP using settings in config/constants.php</li>';
			$status = FALSE;
		}
		else {
			//check admin ldap bind
			$ldap_bind = @ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
			if(FALSE === $ldap_bind) {
				$output .= '<li>Failed to bind to LDAP as '. LDAP_ANON_ADMIN_USERNAME.': ';
				$output .= ldap_error($ldap_conn) .'</li>';
				$status = FALSE;
			}
			$ldap_bind = @ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
			if(FALSE === $ldap_bind) {
				$output .= '<li>Failed to bind to LDAP as '. LDAP_ANON_SEARCH_USERNAME.': ';
				$output .= ldap_error($ldap_conn) .'</li>';
				$status = FALSE;
			}	
		}
		ldap_close($ldap_conn);
		return array('message'=>$output,'status'=>$status);
	 }
	 
	 /* This function checks the datbase configuration and gives a message about it's status
	  * as well as a boolean value for configuration status in the form of an associative array.
	  */
	 private function check_db_configuration() {
		$status = TRUE;
		$output = '';
		//test database connection
		$db = $this->load->database('default',TRUE);
		if(FALSE === $db->conn_id) {
			$output .= '<div class="message_error"><ul>';
			$error_output = '';
			$errors = sqlsrv_errors();
			foreach($errors as $error) { $error_output .= $error['message']; }
			preg_match_all("/Cannot open database \"(.*)?\" requested by the login/",$error_output,$matches);
			if($matches[1][0] === DATABASE_NAME) {
				$output .= '<li>Configured database "'.DATABASE_NAME . '" does not exist yet.</li>';
			}
			$status = FALSE;
		}
		else {
			$output .= '<div class="message_success"><ul>';
			$output .= '<li>Configured database "'.DATABASE_NAME . '" exists.</li></ul></div>';
			$tables = array('account_request','application','application_request','event_log','request','users','logins','mail_log', 'ticket_category', 'tickets');
			$table_errors = ''; $table_success = '';
			foreach($tables as $table) {
				$tbl_check = $db->query('SELECT * FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_TYPE='.$db->escape('BASE TABLE').' AND TABLE_NAME='.$db->escape($table));
				if($tbl_check) {
					if($tbl_check->num_rows() <= 0) { 
						$table_errors .= '<li>Table ' . $table . ' does not exist yet.</li>'; 
					}
					else {
						$table_success .= '<li>Table ' . $table . ' exists.</li>'; 
					}
				}
				else {
					$table_errors .= '<li>Could not get status for table ' . $table . '. Possible permissions issue.</li>'; 
				}
			}
			if(strlen($table_success) > 0) {
				$output .= '<div class="message_success"><ul>';
				$output .= $table_success;
				$output .= '</ul></div>';
			}
			
			if(strlen($table_errors) > 0) {
				$output .= '<div class="message_error"><ul>';
				$output .= $table_errors;
				$status = FALSE;
			}
		}
		return array('message'=>$output .'</ul></div>','status'=>$status);
	 }
	 
	/* This function checks if LDAP is working properly 
	 */
	 private function check_ldap_configuration() {
		$output = '';
		$status = TRUE;
		//if we get to this point, the configuration should be set enough to do a bind as the admin user
		$ldap_conn = $this->prepare_ldap_conn();
		$ldap_bind = @ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if($ldap_bind) {
			$error_output = '';
			$success_output = '';
			//check LDAP structure exists
			$entries = array(
				LDAP_BASE_RDN,
				LDAP_API_ADMIN_GROUP,
				LDAP_APPLICATION_GROUP,
				LDAP_ACCOUNT_GROUP,
				LDAP_DISABLED_APPLICATION_GROUP,
				LDAP_DISABLED_ACCOUNT_GROUP,
				LDAP_DISABLED_GROUPS_GROUP,
				LDAP_GROUPS_GROUP,
				LDAP_DIRECT_API_PERMISSIONS_GROUP,
				LDAP_ADMIN_API_PERMISSIONS_GROUP,
			);
			foreach($entries as $entry) {
				if(!$this->ldap_entry_exists($ldap_conn,$entry)) { $error_output .= '<li>LDAP Entry: ' . $entry . ' does not exist.</li>'; }
				else { $success_output .= '<li>LDAP Entry: ' . $entry . ' exists.</li>'; }
			}
		}
		else {
			$this->session->set_flashdata('errors',ldap_error());
		}
		if(strlen($success_output) > 0) {
			$output .= '<div class="message_success"><ul>';
			$output .= $success_output .'</ul></div>';
		}
		if(strlen($error_output) > 0) {
			$output .= '<div class="message_error"><ul>';
			$output .= $error_output;
			$status = FALSE;
		}
		return array('message'=>$output.'</ul></div>','status'=>$status);
	 }
	
	/* This function checks if the initial user has been set up correctly
	 */
	private function check_user_configuration() {
		$output = '';
		$error_output = '';
		$status = TRUE;
		$org_id = $this->encrypt->decode($this->session->userdata('user_id'));
		$this->load->model('usersmodel');
		if($this->usersmodel->get_user_id_from_org_id($org_id) === FALSE) {
			$error_output .= '<li>User does not exist in database.</li>';
		}
		$ldap_conn = $this->prepare_ldap_conn();
		if(@$this->usersmodel->ldap_bind_current_user($ldap_conn) === FALSE) {
			$error_output .= '<li>Unable to bind to user in LDAP.</li>';
		}
		if(strlen($error_output) > 0) {
			$output .= '<div class="message_error"><ul>';
			$output .= $error_output;
			$status = FALSE;
		}
		return array('message'=>$output.'</ul></div>','status'=>$status);
	}
	 
	/* This function prepares a connection to LDAP using the configured constants for the application
	 * and the LDAP options required for the connection. Returns FALSE on failure, LDAP connection resource
	 * on success.
	 */
	private function prepare_ldap_conn() {
		$ldap_conn = ldap_connect(LDAP_HOSTNAME, LDAP_PORT);
		if(!ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3)) { return FALSE; } 
		if(!ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0)) { return FALSE; }
		return $ldap_conn;
	}
	
	/* This function determines if a given dn exists in ldap by performing an LDAP search and
	 * returning true if the count of results is greater than 0
	 */
	private function ldap_entry_exists($ldap_conn,$dn) {
		$search = @ldap_search($ldap_conn, $dn, '(objectClass=*)');
		$result = @ldap_get_entries($ldap_conn,$search);
		if($result['count'] > 0) { return TRUE; }
		return FALSE;
	}
	 
	private function get_new_username($first,$last) {
		$this->load->model('usersmodel');
		//sanitize username input to remove any non-alphanumeric characters
		$first = preg_replace("/[^A-Za-z0-9]/", '', $first);
		$last = preg_replace("/[^A-Za-z0-9]/", '', $last);
		//theoretically we could hit max username lengths for last names, so don't allow more than 100 characters total
		$last = substr($last,0,99);
		
		if((isset($first) && isset($last)) && (strlen($first) > 0 && strlen($last) > 0)) {
			$start = strtolower($first[0] . $last);
			if($this->usersmodel->username_exists($start)) {
				$i = 2;
				while($this->usersmodel->username_exists($start . $i) && $i < 999) {
					$i++;
				}
				if($i <= 999) { return $start . $i; }
				else { return FALSE; } //returns FALSE if more than 999 users have the same combo of first initial / last name
			}
			else { return $start; }
		}
		else { return FALSE; }
	}
	
}